// Paste Spline.js
//
// 2012-09-29: firset release
//
// Hiroto Tsubaki tg@tres-graficos.jp
//

function buildUI( tool ) {
  tool.addParameterSeparator("Paste Spline");
  
  tool.addParameterLink("pasted spline", false);
  
  tool.addParameterSelector("direction coord.", ["object", "world"], false, false);
  tool.addParameterSelector("paste direction", ["x", "y", "z"], false, false);
  
  tool.addParameterBool("only selected polygon", 0, 0, 1, false, false);
  tool.addParameterBool("edge detection", 1, 0, 1, false, false);
  
  tool.addParameterButton("paste spline", "paste", "pasteSpline");
  
}

function pasteSpline( tool ) {
  var doc = tool.document();
  var obj = doc.selectedObject();
  var spline = tool.getParameter("pasted spline");
  
  if (!obj || !spline) {
    OS.beep(); return;
  }
  
  if (obj.type() == POLYGONOBJ && spline.family() == SPLINEFAMILY) {
    var pi, ei, i, j, k;
    
    var core = obj.core();
    var objMat = obj.obj2WorldMatrix();
    var objMatInverse = objMat.inverse();
    
    var splineCore = spline.core();
    var splineRot = spline.getParameter("rotation");
    var splineRotMat = new Mat4D( ROTATE_HPB, splineRot.x, splineRot.y, splineRot.z );
    var splineMat = spline.obj2WorldMatrix();
    
    //print('----');
    
    var dir = tool.getParameter("paste direction");
    var coord = tool.getParameter("direction coord.");
    var detectEdge = tool.getParameter("edge detection");
    var polysel = tool.getParameter("only selected polygon");
    
    var dirVec;
    switch( parseInt(dir) ) {
      case 0: // -x
        dirVec = new Vec3D( -1, 0, 0 );
        break;
      case 1: // -y
        dirVec = new Vec3D( 0, -1, 0 );
        break;
      case 2: // -z
        dirVec = new Vec3D( 0, 0, -1 );
        break;
    }
    // spline vector
    if (coord == 0) {
      dirVec = splineRotMat.multiply( dirVec );
    }
    dirVec = Vec3D_normalize( dirVec );
    
    var points = new Array();
    var edges = new Array();
    var pathCount = splineCore.pathCount();
    for (i = 0;i < pathCount;i++) {
      var cache = splineCore.cache( i );
      var cacheLen = cache.length;
      for (j = 0;j < cacheLen;j++) {
        points.pushUnique( splineMat.multiply( cache[j] ) );
      }
      for (j = 0;j < cacheLen-1;j++) {
        edges.push( [ splineMat.multiply( cache[j] ), splineMat.multiply( cache[j + 1] ) ] );
      }
    }
    // no spline points
    if (points.length < 1) {
      return 0;
    }
    
    var polygonCount = core.polygonCount();
    
    // for undo/redo
    obj.recordGeometryForUndo();
    
    var crossCountTotal = 0;
    var len = points.length;
    var edgesLen = edges.length;
    
    var raySize = 1;
    
    if (detectEdge) {
      var vertexCount = core.vertexCount();
      var maxVec = new Vec3D();
      var minVec = new Vec3D();
      for (i = 0;i < vertexCount;i++) {
        var vec = core.vertex( i );
        maxVec.x = ( vec.x < maxVec.x )? maxVec.x : vec.x;
        maxVec.y = ( vec.y < maxVec.y )? maxVec.y : vec.y;
        maxVec.z = ( vec.z < maxVec.z )? maxVec.z : vec.z;
        minVec.x = ( vec.x > minVec.x )? minVec.x : vec.x;
        minVec.y = ( vec.y > minVec.y )? minVec.y : vec.y;
        minVec.z = ( vec.z > minVec.z )? minVec.z : vec.z;
      }
    }
    
    for (i = 0;i < polygonCount;i++) {
      var polygonSize = core.polygonSize( i );
      var crossCount = 0;
      
      if (!polysel || core.polygonSelection( i ) ) { 
        for (j = 0;j < polygonSize-2;j++) {
          var triangles = core.triangle(i, j);
          var t0 = core.vertex( core.vertexIndex( i, triangles[0] ) );
          var t1 = core.vertex( core.vertexIndex( i, triangles[1] ) );
          var t2 = core.vertex( core.vertexIndex( i, triangles[2] ) );
            
          t0 = objMat.multiply( t0 );
          t1 = objMat.multiply( t1 );
          t2 = objMat.multiply( t2 );
          
          for (pi = 0;pi < len;pi++) {
            var point = points[pi];
            
            var crossPoint = triangleIntersect( point, dirVec, t0, t1, t2 );
            
            if (crossPoint) {
              var vindex = core.addVertex( false, objMatInverse.multiply( crossPoint ) );
              core.setVertexSelection( vindex, true );
              crossCount++;
            }
          }
        }
        
        if (detectEdge) {
          for (j = 0;j <polygonSize;j++) {
            var eIndex0 = j;
            var eIndex1 = (polygonSize == j + 1)? 0 : j + 1;
            
            var e0 = objMat.multiply( core.vertex( core.vertexIndex( i, eIndex0 ) ) );
            var e1 = objMat.multiply( core.vertex( core.vertexIndex( i, eIndex1 ) ) );
            
            var eDist = Vec3D_distance( e0, e1 );
            var eDirVec = Vec3D_normalize( e0.sub( e1 ) );
            
            var objSize = Vec3D_distance( minVec, maxVec );
            
            for (ei = 0;ei < edgesLen;ei++) {
              var edge = edges[ei];
              var dist = Vec3D_distance( e0, edge[0] ) + objSize;
              
              var p0 = edge[0].add( dirVec.multiply( dist ) );
              var p1 = edge[1].add( dirVec.multiply( dist ) );
              var p2 = edge[0].sub( dirVec.multiply( dist ) );
              var p3 = edge[1].sub( dirVec.multiply( dist ) );
              
              //core.addPolygon( 3, false, [p0, p1, p2] );
              //core.addPolygon( 3, false, [p3, p2, p1] );
              
              var crossPoint = triangleIntersect( e0, eDirVec, p0, p1, p2 );
              if (!crossPoint) {
                crossPoint = triangleIntersect( e0, eDirVec, p3, p2, p1 );
              }
              
              if (crossPoint) {
                
                if ( Vec3D_distance( e0, crossPoint ) < eDist 
                      && Vec3D_distance( e1, crossPoint) < eDist ) {
                  var vindex = core.addVertex( false, objMatInverse.multiply( crossPoint ) );
                  core.setVertexSelection( vindex, true );
                  crossCount++;
                }
                
              }
            }
          }
        }
      }
      crossCountTotal += crossCount;
    }
    
    //print( 'totalPoint:'+crossCountTotal );
    
    obj.update();
  }
}

// Tomas Möller 
var triangleIntersect = function( orig, dir, v0, v1, v2 ) {
  var e1, e2, pvec, tvec, qvec;
  var epsilon = 1e-6;
  var det;
  var t, u, v;
  var inv_det;
  
  e1 = v1.sub( v0 );
  e2 = v2.sub( v0 );
  
  pvec = dir.cross( e2 );
  det = e1.dot( pvec );
  
  if (det.x > epsilon) {
    tvec = orig.sub( v0 );
    u = tvec.dot( pvec );
    if (u.x < 0 || u.x > det.x) return false;
    
    qvec = tvec.cross( e1 );
    
    v = dir.dot( qvec );
    if (v.x < 0 || u.x + v.x > det.x) return false; 
  } else if (det.x < -epsilon) {
    tvec = orig.sub( v0 );
    
    u = tvec.dot( pvec );
    if (u.x > 0 || u.x < det.x) return false;
    
    qvec = tvec.cross( e1 );
    
    v = dir.dot( qvec );
    if (v.x > 0 || u.x + v.x < det.x) return false;
  } else {
    return false;
  }
  
  inv_det = 1 / det.x;
  
  t = e2.dot( qvec );
  t = t.x * inv_det;
  
  u = u.x * inv_det;
  v = v.x * inv_det;
  
  var crossPoint = orig.add( dir.multiply( t ) );
  
  return crossPoint;
}

var Vec3D_distance = function() {
  if( arguments.length == 1)
      return Math.sqrt( arguments[0].x*arguments[0].x + arguments[0].y*arguments[0].y + arguments[0].z*arguments[0].z );
  var p = arguments[1].sub(arguments[0]);
  return Math.sqrt( p.x*p.x + p.y*p.y + p.z*p.z );
}

var Vec3D_normalize = function(vec) {
  var l = vec.norm();
  if (l != 0) {
    return vec.multiply( 1/l );
  }
  return vec;
}

Array.prototype.pushUnique = function( vec ) {
  var len = this.length;
  for (var i = 0;i < len;i++) {
    var data = this[i];
    if (data.isEqual( vec )) {
      return i;
    }
  }
  this.push( vec );
}